unit OOoTools;

interface

uses Variants;

var
  OpenOffice, StarDesktop: Variant;
  OOoIntrospection, OOoReflection: Variant;

procedure ConnectOpenOffice;
function  CreateUnoService(serviceName: String): Variant;
function  CreateUnoStruct(structName: String; indexMax: Integer= -1): Variant;
function  CreateProperties(propertyList: array of Variant): Variant;
function  MakePropertyValue(PropName: string; PropValue: Variant): Variant;
function  HasUnoInterfaces(thisObject: Variant; interfaceList: array of String): Boolean;
function  isNullEmpty(thisVariant: Variant): Boolean;
function  dummyArray: Variant;
function  GetProcessServiceManager: Variant;

procedure execDispatch(Command: String; params: Variant);
procedure copyToClipboard;
procedure pasteFromClipboard;
function  convertToURL(winAddr: String): String;
function  convertFromURL(URLaddr: String): String;
function  RGB(redV, greenV, blueV: byte): Longword;
function  Red(colorOOo: Longword): Byte;
function  Green(colorOOo: Longword): Byte;
function  Blue(colorOOo: Longword): Byte;

{
This unit is part of a toolbox to pilot OpenOffice.org from Delphi using COM Automation.
Copyright (C) 2004  Bernard Marcelly
This unit is free software; you can redistribute it and/or modify it under the terms of
the GNU Lesser General Public License as published by the Free Software Foundation;
either version 2.1 of the License, or (at your option) any later version.
This library is distributed in the hope that it will be useful,but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU Lesser General Public License for more details.
You should have received a copy of the GNU Lesser General Public License along with this
library; if not, write to the Free Software Foundation, Inc.,
59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}

implementation

uses Classes, Controls, Forms, SysUtils, ComObj;

type
  EOOoError= class(Exception);

const USASCIIexcl =   // caractres US-ASCII  remplacer par % hexa
  '%%25 %20<%3C>%3E#%23"%22{%7B}%7D|%7C\%5C^%5E[%5B]%5D`%60';
const UTF8chars =        // caractres non US-ASCII  remplacer par % hexa % hexa
  '%C3%80%C3%81%C3%82%C3%83%C3%84%C3%85%C3%86%C3%87%C3%88%C3%89%C3%8A%C3%8B%C3%8C%C3%8D%C3%8E%C3%8F' +
  '%C3%90%C3%91%C3%92%C3%93%C3%94%C3%95%C3%96%C3%97%C3%98%C3%99%C3%9A%C3%9B%C3%9C%C3%9D%C3%9E%C3%9F' +
  '%C3%A0%C3%A1%C3%A2%C3%A3%C3%A4%C3%A5%C3%A6%C3%A7%C3%A8%C3%A9%C3%AA%C3%AB%C3%AC%C3%AD%C3%AE%C3%AF' +
  '%C3%B0%C3%B1%C3%B2%C3%B3%C3%B4%C3%B5%C3%B6%C3%B7%C3%B8%C3%B9%C3%BA%C3%BB%C3%BC%C3%BD%C3%BE%C3%BF' +
  '%C5%92%C5%93';

const URLprefix : Array [1..7] of String =
    ('file:', 'ftp:', 'news:', 'http:', 'mailto:', 'macro:', 'private:');

const  // textes des messages d'erreurs
  mess01= 'Connexion OpenOffice impossible';
  mess02= 'Liste d''arguments incorrecte';
  mess03= 'L''argument de rang %d ( partir de 0) est incorrect';
  mess05= 'Impossible de crer le service : %s';
  mess06= 'La fentre courante n''est pas un document OpenOffice';
  mess07= 'Impossible de connatre les interfaces de cet objet';


var
  disp : Variant;

  { -------------------------------------------------- }


// vrifie si le Variant contient bien quelque chose
function  isNullEmpty(thisVariant: Variant): Boolean;
begin
  Result:= VarIsEmpty(thisVariant) or VarIsNull(thisVariant) or VarIsClear(thisVariant);
end;




// quivalent de la fonction OOoBasic
function CreateUnoService(serviceName: String): Variant;
begin
  Result:= OpenOffice.createInstance(serviceName);
  if isNullEmpty(Result) then    Raise EOOoError.Create(Format(mess05, [serviceName]));
end;



// quivalent de la fonction OOoBasic; normalement inutile pour l'utilisateur
function GetProcessServiceManager: Variant;
begin
  Result := CreateOleObject('com.sun.star.ServiceManager');
  if isNullEmpty(Result) then    Raise EOOoError.Create(mess01);
end;


// initialiser l'interface vers OpenOffice
procedure ConnectOpenOffice;
begin
  if not isNullEmpty(OpenOffice) then exit;
  Screen.Cursor:= crHourglass;      Application.ProcessMessages;
  try
    OpenOffice:= GetProcessServiceManager;
    StarDesktop:=       CreateUnoService('com.sun.star.frame.Desktop');
    disp:=              CreateUnoService('com.sun.star.frame.DispatchHelper');
    OOoIntrospection:=  CreateUnoService('com.sun.star.beans.Introspection');
    OOoReflection:=     CreateUnoService('com.sun.star.reflection.CoreReflection');
  finally
    Screen.Cursor:= crDefault;
  end;
end;


// quivalent de la fonction OOoBasic
function CreateUnoStruct(structName: String; indexMax: Integer= -1): Variant;
var
  d: Integer;
begin
  if indexMax < 0  then
    Result:= OpenOffice.Bridge_GetStruct(structName)
  else begin
    Result:= VarArrayCreate([0, indexMax], varVariant);
    for d:= 0 to indexMax  do
      Result[d]:=  OpenOffice.Bridge_GetStruct(structName);
  end;
end;


// quivalent de la fonction de Danny Brewer
function MakePropertyValue(PropName: string; PropValue: Variant): Variant;
begin
  Result:= OpenOffice.Bridge_GetStruct('com.sun.star.beans.PropertyValue');
  Result.Name:= PropName;   Result.Value:= PropValue;
end;


// fonction plus puisssante que MakePropertyValue
function CreateProperties(propertyList: array of Variant): Variant;
var
  x, y, xMax: Integer;
begin
  xMax:= High(propertyList);
  if (not odd(xMax)) or (xMax < 1)  then
    Raise EOOoError.Create(mess02);

  Result:= VarArrayCreate([0, xMax shr 1], varVariant);   x:= 0;  y:= 0;
  repeat
    Result[y]:=  OpenOffice.Bridge_GetStruct('com.sun.star.beans.PropertyValue');
    Case VarType(propertyList[x])of { vrifier que l'argument est bien un String }
    varOleStr, varStrArg, varString:    Result[y].Name:= propertyList[x];
    else
      Raise EOOoError.Create(Format(mess03, [x]));
    end;
    Result[y].Value:= propertyList[x +1];
    inc(y); inc(x,2);
  until x > xMax;
end;


// cre un tableau vide, pour une liste vide
function dummyArray: Variant;
begin
  Result:= VarArrayCreate([0, -1], varVariant);
end;



// quivalent de la fonction OOoBasic
function  HasUnoInterfaces(thisObject: Variant; interfaceList: array of String): Boolean;
var
  objInterf: TStringList;
  insp, info1, info2, info3: Variant;  x, x2 : Integer;  oneInterf: String;
begin
  Result:= False;
  objInterf:= TStringList.Create;
  try
    insp:= OOoIntrospection.inspect(thisObject);
    info1:= insp.getMethods(-1);
    for x:= 0 to VarArrayHighBound(info1, 1) do begin
      info2:= info1[x];  info3:= info2.DeclaringClass;  oneInterf:= info3.Name;
      if (oneInterf <> '')  and (objInterf.IndexOf(oneInterf) < 0)  then
        objInterf.Add(oneInterf);
    end;
    for x:= 0 to High(interfaceList) do begin
      x2:= objInterf.IndexOf(interfaceList[x]);
      if x2 < 0  then exit;
      if objInterf.Strings[x2] <> interfaceList[x]  then exit; // vrifier la casse
    end;
    Result:= True;
  except
    Raise EOOoError.Create(mess07);
  end;
end;


// appel du dispatch OpenOffice, comme l'enregistreur de macro
procedure execDispatch(Command: String; params: Variant);
begin
  disp.executeDispatch(StarDesktop.CurrentFrame, Command, '', 0, params);
end;


procedure copyToClipboard;
begin
  execDispatch('.uno:Copy', dummyArray);
end;


procedure pasteFromClipboard;
begin
  execDispatch('.uno:Paste', dummyArray);
end;




{ ---------  conversion d'URL : voir RFC 2396  ---------------------
  les caractres  transformer sont convertis en UTF8,
  et chaque octet est crit en hexadecimal prcd de %

  fonctions similaires  celles de OOoBasic    }

function convertToURL(winAddr: String): String;
var
  x : Integer; s, sLow : String;

  function escapeToUTF8URL(c: Char): String;
  var
    x: Integer;
  begin
    if ord(c) < 128  then begin
      x:= Pos(c, USASCIIexcl);
      if (c in ['0'..'9', 'a'..'z', 'A'..'Z']) or (x = 0) then  
        Result:= c    // caractres accepts, rservs ou non rservs
      else            // caractres exclus,  convertir
        Result:= Copy(USASCIIexcl, x+1, 3);
    end else begin    // conversion en UTF8  deux octets
      x:= Pos(c, UTF8chars);
      if x > 0 then
        Result:= Copy(UTF8chars, x+1, 6)
      else            // caractre inconnu ( mettre  jour la table UTF8chars ? )
        Result:= '?';
    end;
  end;

  function existsPrefix: Boolean;
  var
    x: Integer;
  begin
    Result:= False;
    for x:= 1 to High(URLprefix) do begin
      if Pos(URLprefix[x], sLow) = 1 then
        begin Result:= True; break; end;
    end;
  end;

begin { -------- convertToURL ---------- }
  s:= StringReplace(winAddr, '\', '/', [rfReplaceAll]);
  sLow:= AnsiLowerCase(s);
  if existsPrefix then
    Result:= ''
  else
    if Pos('@', sLow) > 0 then
      Result:= 'mailto:'
    else
      Result:= 'file:///';
  for x:= 1 to Length(s) do
    Result:= Result + escapeToUTF8URL(s[x]);
end;


function convertFromURL(URLaddr: String): String;
const
  pr= 'file:///';
var
  s : String; x: Integer;
begin
  s:= URLaddr;
  if Pos(pr, LowerCase(URLaddr)) = 1 then begin   // cas particulier : file:///
    Delete(s, 1, Length(pr));
    s:= StringReplace(s, '/', '\', [rfReplaceAll]);
  end;
  x:= 1;
  while x < Length(UTF8chars) do begin  // remplacer d'abord les codes  2 octets
    s:= StringReplace(s, Copy(UTF8chars, x+1, 6), UTF8chars[x], [rfReplaceAll, rfIgnoreCase]);
    inc(x, 7);
  end;
  x:= 1;
  while x < Length(USASCIIexcl) do begin
    s:= StringReplace(s, Copy(USASCIIexcl, x+1, 3), USASCIIexcl[x], [rfReplaceAll, rfIgnoreCase]);
    inc(x, 4);
  end;
  Result:= s;
end;


{  -------------  fonctions de couleurs  ---------------
        fonctions identiques  celles de OOoBasic               }


function RGB(redV, greenV, blueV: byte): Longword;
begin
  Result:= (redV shl 16) + (greenV shl 8) +blueV
end;


function Blue(colorOOo: Longword): Byte;
begin
  Result:= colorOOo and 255
end;


function Green(colorOOo: Longword): Byte;
begin
  Result:= (colorOOo shr 8) and 255
end;


function Red(colorOOo: Longword): Byte;
begin
  Result:= (colorOOo shr 16) and 255
end;

end.
